/*= GM SSP X Power Source (VISA) ============================================*/
/* LabWindows/CVI 5.0.1 VXI PnP Instrument Driver                            */
/* Original Release: October 1999                                            */
/* By: Jiri Hula, ELCOM,a.s., Czech Republic                                 */
/*     PH.  +420 69 699 6146      Fax +420 69 699 6147                       */
/*     Email: jiri.hula@elcom.cz                                             */
/*                                                                           */
/* Modification History:                                                     */
/*																			 */
/* July 2000: Rev 1.1, 07/2000, CVI 5.5                                      */
/*            - error defines added (QUERY_ERR, EXEC_ERR, ....)              */
/*            - gmsspx_error_query changed (returns not supported)			 */
/*            - check_error changed                                          */
/*            - added 2ms delay for I/O operations                           */
/*            By: Marek Malohlava, ELCOM, a.s., Czech Republic               */
/*                                                                           */
/*===========================================================================*/

#include <utility.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <visa.h>
#include <ctype.h>
#include "gmsspx.h"


#define GMSSPX_REVISION     "Rev 1.1, 07/2000, CVI 5.5" /*  Instrument driver revision */
#define BUFFER_SIZE         512L         /* File I/O buffer size */
#define DEFAULT_IO_DELAY  0.002

/*= Macros ==================================================================*/
#define CHECKERR(fCal) if (gmsspx_status = (fCal), gmsspx_status < VI_SUCCESS) \
              return gmsspx_status; else

/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
 static ViString StateArr[] = { "OFF", "ON", VI_NULL};
 static ViString PowerStateArr[] = { "RST", "RCL", "SBY", VI_NULL};
 static ViString SequenceTriggArr[] = { "GO", "HOLD", "CONT", "STRT", "STEP",
                    "STOP"};
 static ViString Signal12StateArr[] = { "OFF", "ON", "OUT", "MODE", "SEQ", "SSET", "U_LO",
                      "U_HI", "I_LO", "I_HI", VI_NULL};
 static ViString TriggerModeStateArr[] = { "OFF", "OUT", "RCL", "SEQ", "LLO", "MIN", VI_NULL};
 static ViString InstrumentModelArr[] = { "32N020RU010", "32N040RU006", "32N080RU003",
                      "32N020RU020", "32N040RU012", "32N080RU006",
                      "32N032RU018", "62N040RU025", "62N040RU050",
                      "64N040RU100", "64N040RU150", "62N080RU125",
                      "62N080RU025", "64N080RU050", "64N080RU075",
                      "62N052RU025", "62N052RU050", "64N052RU100",
                      "64N052RU150", VI_NULL};
 static ViReal64 CurrentMaxArr[] = { 10.0, 6.0, 3.0, 20.0, 12.0, 6.0, 18.0, 25.0,
                   50.0, 100.0, 150.0, 12.5, 25.0, 50.0, 75.0,
                   25.0, 50.0, 100.0, 150.0};
 static ViReal64 VoltageMaxArr[] = { 20.0, 40.0, 80.0, 20.0, 40.0, 80.0, 32.0,
                   40.0, 40.0, 40.0, 40.0, 80.0, 80.0, 80.0,
                   80.0, 52.0, 52.0, 52.0, 52.0};
 static ViString OutputModeArr[] = { "OFF", "CV", "CC", "OL", VI_NULL};
 static ViString MinMaxModeArr[] = { "OFF", "ON", "RST", VI_NULL}; 
 static ViString PowerModeArr[] = { "RST", "RCL", "SBY", VI_NULL}; 
 static ViString SeqModeArr[] = { "GO", "ON", "OFF", "HOLD", "CONT", "STRT", "STEP",
                  "STOP", VI_NULL};
 static ViString Signal12Arr[] = { "OFF", "ON", "OUT", "MODE", "SEQ", "SSET", "U_LO",
                   "U_HI", "I_LO", "I_HI", VI_NULL};
 static ViString TriggerModeArr[] = { "OFF", "OUT", "RCL", "SEQ", "LLO", "MIN", VI_NULL};
 static ViString StoreStateArr[] = { "OFF", "ON", "CLR", VI_NULL}; 
 static ViString SetRegArr[] = { "*ESE", "ERAE", "ERBE", "*SRE", "*PRE"};
 static ViString GetRegArr[] = { "*ESR?", "*ESE?", "CRA?", "ERA?", "ERAE?", "ERB?", "ERBE?",
                 "*STB?", "*SRE?", "*PRE?"};
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* gmsspx_stringValPair is used in the gmsspx_errorMessage function          */
/* gmsspx_statusDataRanges is used to track session dependent status & ranges*/
/*===========================================================================*/
typedef struct  gmsspx_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  gmsspx_tStringValPair;
 
struct gmsspx_statusDataRanges {
    ViBoolean errorChecking;
    
    ViInt32 instrModel;
    ViChar instrDriverRevision[BUFFER_SIZE];
    ViReal64 ioDelay;
};

typedef struct gmsspx_statusDataRanges *gmsspx_instrRange;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean gmsspx_invalidViBooleanRange (ViBoolean val);
ViBoolean gmsspx_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean gmsspx_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViBoolean gmsspx_invalidViString (ViString val, ViUInt32 maxLength);
ViStatus gmsspx_waitOnVisaEvent (ViSession instrSession, ViEventType eventTypeIn,
  ViUInt32 tmoIn, ViPUInt16 STB);
ViStatus gmsspx_initCleanUp (ViSession openRMSession, ViPSession openInstrSession,
  ViStatus currentStatus);
ViStatus gmsspx_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*------ INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINE DECLARATIONS HERE ------*/
/*****************************************************************************/
ViStatus gmsspx_checkStatus (ViSession instrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/
/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_init (ViRsrc resourceName, ViBoolean IDQuery,
                    ViBoolean resetDevice, ViPSession instrSession)
{
    ViStatus  gmsspx_status = VI_SUCCESS;
    ViSession rmSession = 0;
    ViByte    rdBuffer[BUFFER_SIZE];
    ViUInt16  intfType;
    ViUInt32 retCnt;
    gmsspx_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/
    if (gmsspx_invalidViBooleanRange (IDQuery))
        return VI_ERROR_PARAMETER2;
    if (gmsspx_invalidViBooleanRange (resetDevice))
        return VI_ERROR_PARAMETER3;

    /*- Open instrument session ---------------------------------------------*/
    if ((gmsspx_status = viOpenDefaultRM (&rmSession)) < 0)
        return gmsspx_status;

    if ((gmsspx_status = viOpen (rmSession, resourceName, VI_LOAD_CONFIG,
    VI_NULL, instrSession)) < 0) {
        viClose (rmSession);
        return gmsspx_status;
    }

    instrPtr = malloc (sizeof (struct gmsspx_statusDataRanges));
    instrPtr->ioDelay = DEFAULT_IO_DELAY; // 2 ms
    CHECKERR (viSetAttribute (*instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr));
    
    /*- Configure VISA Formatted I/O ----------------------------------------*/
    
    if ((gmsspx_status = viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &intfType)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
            
    if (intfType == VI_INTF_ASRL)
      {
      if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\n')) < 0)
        return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_TRUE)) < 0)
      return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
      return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_TERMCHAR)) < 0)
      return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
        }    
    
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 10000)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    
    if ((gmsspx_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
  
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    
    if ((gmsspx_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);

    /*- Identification Query ------------------------------------------------*/
    if (IDQuery) {
        
        Delay (instrPtr->ioDelay);
        if ((gmsspx_status = viWrite (*instrSession, "*IDN?", 5, &retCnt)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);

        Delay (instrPtr->ioDelay);
        if ((gmsspx_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);

        if (strncmp ((ViChar *)rdBuffer, "GOSSEN", 6))
          return gmsspx_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
    }     

    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice) {
      if ((gmsspx_status = gmsspx_reset (*instrSession)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    } else { /*- Send Default Instrument Setup ---------------------------------*/
      if ((gmsspx_status = gmsspx_defaultInstrSetup (*instrSession)) < 0)
            return gmsspx_initCleanUp (rmSession, instrSession, gmsspx_status);
    }
          
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Aplication Example                                              */
/* Purpose:  This function sets current, voltage, current & voltage limit  */
/*           and sets output to On state                   */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_AppExample (ViSession instrSession, ViReal64 voltageLimit,
                                     ViReal64 currentLimit, ViReal64 voltage,
                                   ViReal64 current, ViPReal64 nominalVoltage,
                                   ViPReal64 nominalCurrent)
{
  ViStatus gmsspx_status = VI_SUCCESS; 
  gmsspx_instrRange instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViReal64Range (voltageLimit, 0.0, VoltageMaxArr[instrPtr -> instrModel]))  
       return VI_ERROR_PARAMETER2;
    
    if (gmsspx_invalidViReal64Range (currentLimit, 0.0, CurrentMaxArr[instrPtr -> instrModel]))  
       return VI_ERROR_PARAMETER3;
    
    if (gmsspx_invalidViReal64Range (voltage, 0.0, VoltageMaxArr[instrPtr -> instrModel]))   
       return VI_ERROR_PARAMETER4;
 
    if (gmsspx_invalidViReal64Range (current, 0.0, CurrentMaxArr[instrPtr -> instrModel]))   
       return VI_ERROR_PARAMETER5;
    
    if (!nominalVoltage)   
       return VI_ERROR_PARAMETER6;
    
    if (!nominalCurrent)   
       return VI_ERROR_PARAMETER7;
    
  CHECKERR (gmsspx_confSetVoltLimit (instrSession, voltageLimit));
  CHECKERR (gmsspx_confSetCurrLimit (instrSession, currentLimit));
  CHECKERR (gmsspx_confSetVolt (instrSession, voltage));
  CHECKERR (gmsspx_confSetCurrent (instrSession, current));
  CHECKERR (gmsspx_confSetOutputState (instrSession, VI_TRUE));
  CHECKERR (gmsspx_measOutVolt (instrSession, nominalVoltage));
  CHECKERR (gmsspx_measOutCurr (instrSession, nominalCurrent));

  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Voltage Setting                                             */
/* Purpose:  This function sets the setpoint value of the output voltage.    */                                     
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetVolt (ViSession instrSession,
                                      ViReal64 voltage)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
  ViUInt32 cnt, retCnt;
  ViChar buf[BUFFER_SIZE];
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  if (gmsspx_invalidViReal64Range (voltage, 0.0, VoltageMaxArr[instrPtr -> instrModel]))   
    return VI_ERROR_PARAMETER2;

  CHECKERR (viPrintf (instrSession, "US %.3f", voltage));

  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Voltage Setting                                             */
/* Purpose:  This function queries the setpoint value of the output voltage. */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetVolt (ViSession instrSession,
                                      ViPReal64 voltage)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  ViChar buf[BUFFER_SIZE];
  ViUInt32 retCnt;
  gmsspx_instrRange instrPtr;

  if (!voltage)  
       return VI_ERROR_PARAMETER2;
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

  Delay (instrPtr->ioDelay);
  CHECKERR (viWrite (instrSession, "US?", 3, &retCnt));
  Delay (instrPtr->ioDelay);
  CHECKERR (viRead (instrSession, buf, BUFFER_SIZE, &retCnt));
  buf[retCnt - 1] = 0;
  if (sscanf (buf, "%*[^ ] %Lf", voltage) != 1)
    return GMSSPX_ERROR_INSTRUMENT_ERROR;
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Voltage Limit                                               */
/* Purpose:  This function defines the upper soft limit for the voltage		 */
/*     		 setpoint.   										             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetVoltLimit (ViSession instrSession,
                                           ViReal64 voltageLimit)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  ViUInt32 cnt, retCnt;
  ViChar buf[BUFFER_SIZE];
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  if (gmsspx_invalidViReal64Range (voltageLimit, 0.0, VoltageMaxArr[instrPtr -> instrModel]))  
       return VI_ERROR_PARAMETER2;
    
  cnt = sprintf (buf, "UL %.3f;", voltageLimit);
  Delay (instrPtr->ioDelay);
  CHECKERR (viWrite (instrSession, buf, cnt, &retCnt));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Voltage Limit                                               */
/* Purpose:  This function queries the upper soft limit for the voltage		 */
/*			 setpoint.														 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetVoltLimit (ViSession instrSession,
                                           ViPReal64 voltageLimit)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
  
  if (!voltageLimit)   
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "UL?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", voltageLimit));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set OVP Level                                                   */
/* Purpose:  This function defines Over Voltage Protection level.            */                   
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetOVPLevel (ViSession instrSession,
                                          ViReal64 level)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
   
  if (gmsspx_invalidViReal64Range (level, 3.0, 1.25*VoltageMaxArr[instrPtr -> instrModel]))  
    return VI_ERROR_PARAMETER2;
    
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OVS %.3f", level));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get OVP Level                                                   */
/* Purpose:  This function queries Over Voltage Protection level.            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetOVPLevel (ViSession instrSession,
                                          ViPReal64 level)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

  if (!level)  
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OVS?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", level));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Get Maximum Voltage                                           */
/* Purpose:  This function returns the upper voltage limit of the 		   */
/*			 instrument.												   */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetVoltMax (ViSession instrSession,
                ViPReal64 maxVoltageinV)
{
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  *maxVoltageinV = VoltageMaxArr[instrPtr -> instrModel];
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Current Setting                                             */
/* Purpose:  This function defines the setpoint value of the output current. */                                     
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetCurrent (ViSession instrSession,
                                         ViReal64 current)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
  ViUInt32 cnt, retCnt;
  ViChar buf[BUFFER_SIZE];
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

  if (gmsspx_invalidViReal64Range (current, 0.0, CurrentMaxArr[instrPtr -> instrModel]))   
    return VI_ERROR_PARAMETER2;
    
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "IS %.3f", current));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Current Setting                                             */
/* Purpose:  This function queries the setpoint value of the output current. */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetCurrent (ViSession instrSession,
                                         ViPReal64 current)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  ViChar buf[BUFFER_SIZE];
  ViUInt32 retCnt;
  gmsspx_instrRange instrPtr;
    
  if (!current)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

  Delay (instrPtr->ioDelay);
  CHECKERR (viWrite (instrSession, "IS?", 3, &retCnt));
  Delay (instrPtr->ioDelay);
  CHECKERR (viRead (instrSession, buf, BUFFER_SIZE, &retCnt));
  buf[retCnt - 1] = 0;
  if (sscanf (buf, "%*[^ ] %Lf", current) != 1)
    return GMSSPX_ERROR_INSTRUMENT_ERROR;
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Current Limit                                               */
/* Purpose:  This function defines the upper soft limit for the current		 */
/*			 setpoint.                           							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetCurrLimit (ViSession instrSession,
                                           ViReal64 currentLimit)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  ViUInt32 retCnt, cnt;
  ViChar buf[BUFFER_SIZE];
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
 
  if (gmsspx_invalidViReal64Range (currentLimit, 0.0, CurrentMaxArr[instrPtr -> instrModel]))  
     return VI_ERROR_PARAMETER2;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "IL %.3f", currentLimit));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Current Limit                                               */
/* Purpose:  This function queries the upper soft limit for the current		 */
/*			 setpoint.  										             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetCurrLimit (ViSession instrSession,
                                           ViPReal64 currentLimit)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
 
  if (!currentLimit)   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "IL?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", currentLimit));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set OCP Delay                                                   */
/* Purpose:  This function defines the OCP delay.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetOCPDelay (ViSession instrSession,
                                          ViReal64 OCPDelay)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

  if (gmsspx_invalidViReal64Range (OCPDelay, 0.0, 99.99))  
    return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay); 
  CHECKERR (viPrintf (instrSession, "DE %.2f", OCPDelay));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get OCP Delay                                                   */
/* Purpose:  This function queries the OCP delay.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetOCPDelay (ViSession instrSession,
                                          ViPReal64 OCPDelay)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  
  if (!OCPDelay)   
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "DE?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", OCPDelay));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set OCP State                                                   */
/* Purpose:  This function selects OCP ON/OFF.                               */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetOCPState (ViSession instrSession,
                                          ViBoolean state)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

  if (gmsspx_invalidViBooleanRange (state))  
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OC %s", StateArr[state]));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get OCP State                                                   */
/* Purpose:  This function queries OCP state.                                */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetOCPState (ViSession instrSession,
                                          ViPBoolean state)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  ViChar mode[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;

  if (!state)  
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OC?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  *state = (ViBoolean) strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE;    
    
  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Get Maximum Current                                           */
/* Purpose:  This function returns the maximum current range value of the  */
/*           connected model of instrument.                                */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetCurrMax (ViSession instrSession,
                ViPReal64 maxCurrentinA)
{
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  *maxCurrentinA = CurrentMaxArr[instrPtr -> instrModel];
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Output State                                                */
/* Purpose:  This function selects power output ON/OFF.                      */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetOutputState (ViSession instrSession,
                                             ViBoolean state)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  ViUInt32 cnt, retCnt;
  ViChar buf[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;
    
  if (gmsspx_invalidViBooleanRange (state))  
    return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OU %s;*WAI", StateArr[state]));

  CHECKERR (gmsspx_checkStatus (instrSession));
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Output State                                                */
/* Purpose:  This function queries power output state.                       */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetOutputState (ViSession instrSession,
                                             ViPBoolean state)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  ViChar mode[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;

  if (!state)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "OU?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  *state = (ViBoolean) strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE;    
   
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Get Output Mode                                                 */
/* Purpose:  This function queries the current operating mode of the output. */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetOutputMode (ViSession instrSession,
                                            ViPInt32 mode)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar smode[BUFFER_SIZE];
    ViInt32 i;
  gmsspx_instrRange instrPtr;
    
    if (!mode)   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "MODE?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", smode));

  for (i = 0; OutputModeArr[i]; i++)
    if (!strcmp(smode, OutputModeArr[i])) break;
  
  if (!OutputModeArr[i])
    return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

    *mode = i;    
    
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Sequence Setting                                                */
/* Purpose:  This function sets the Sequence.        						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeq (ViSession instrSession, ViBoolean mode,
                                     ViInt32 startIndex, ViInt32 stopIndex,
                                     ViInt32 repetition,
                                     ViReal64 time)
{
  ViStatus gmsspx_status = VI_SUCCESS; 
  gmsspx_instrRange instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViBooleanRange (mode))   
       return VI_ERROR_PARAMETER2;
    
    if (gmsspx_invalidViInt32Range (startIndex, 11, 255))  
       return VI_ERROR_PARAMETER3;
    
    if (gmsspx_invalidViInt32Range (startIndex, 11, stopIndex))  
       return VI_ERROR_PARAMETER3;
    
    if (gmsspx_invalidViInt32Range (stopIndex, 11, 255))   
       return VI_ERROR_PARAMETER4;
    
    if (gmsspx_invalidViInt32Range (repetition, 0, 245))   
       return VI_ERROR_PARAMETER5;
 
    if (gmsspx_invalidViReal64Range (time, 0.01, 99.99))   
       return VI_ERROR_PARAMETER6;
    
  CHECKERR (gmsspx_confSetSeqState (instrSession, mode));
  CHECKERR (gmsspx_confSetSeqRange (instrSession, startIndex, stopIndex));
  CHECKERR (gmsspx_confSetSeqRepetition (instrSession, repetition));  
  CHECKERR (gmsspx_confSetSeqDefaultTime (instrSession, time));
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Sequence State                                              */
/* Purpose:  This function sets the Sequence State.                          */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeqState (ViSession instrSession, ViBoolean mode)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViBooleanRange (mode))   
       return VI_ERROR_PARAMETER2;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SEQ %s", StateArr[mode]));
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sequence State                                              */
/* Purpose:  This function queries Sequence State.                           */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSeqState (ViSession instrSession, ViPBoolean sMode)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;
    
  if (!sMode)  
    return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SEQ?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  *sMode = (ViBoolean) strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE;     
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set Sequence Repetition                                         */
/* Purpose:  This function defines Sequence Repetition.                      */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeqRepetition (ViSession instrSession,
                                               ViInt32 sequenceRepetition)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViInt32Range (sequenceRepetition, 0, 255))   
       return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "REPETITION %d", sequenceRepetition));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sequence Repetition                                         */
/* Purpose:  This function queries Sequence Repetition.       				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSeqRepetition (ViSession instrSession,
                                               ViPInt32 sequenceRepetition)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (!sequenceRepetition)   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "REPETITION?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %d", sequenceRepetition));

    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Sequence Start Stop Address                                 */
/* Purpose:  This function defines Sequence Range.                           */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeqRange (ViSession instrSession,
                                          ViInt32 startAddress, ViInt32 stopAddress)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

  if (gmsspx_invalidViInt32Range (stopAddress, 11, 255))   
       return VI_ERROR_PARAMETER3;

  if (gmsspx_invalidViInt32Range (startAddress, 11, stopAddress))  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "STA %d, %d", startAddress, stopAddress));

  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sequence Start Stop Address                                 */
/* Purpose:  This function queries Sequence Range.                           */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSeqRange (ViSession instrSession,
                                          ViPInt32 startAddress, ViPInt32 stopAddress)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (!startAddress)   
       return VI_ERROR_PARAMETER2;

    if (!stopAddress)  
       return VI_ERROR_PARAMETER3;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "STA?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %d,%d", startAddress, stopAddress));
  
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Sequence Default Time                                       */
/* Purpose:  This function defines Default Sequence Time.                    */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeqDefaultTime (ViSession instrSession,
                                            ViReal64 sequenceDefaultTime)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  
    if (gmsspx_invalidViReal64Range (sequenceDefaultTime, 0.01, 99.99))  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "TD %.2f", sequenceDefaultTime));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sequence Default Time                                       */
/* Purpose:  This function queries Default Sequence Time.                    */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSeqDefaultTime (ViSession instrSession,
                                            ViPReal64 sequenceDefaultTime)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (!sequenceDefaultTime)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "TD?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", sequenceDefaultTime));
  
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Sequence Step Time                                          */
/* Purpose:  This function defines Sequence Time.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSeqTime (ViSession instrSession,
                                         ViReal64 sequenceTime)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  
    if (gmsspx_invalidViReal64Range (sequenceTime, 0.0, 99.99))  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "TS %.2f", sequenceTime));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sequence Step Time                                          */
/* Purpose:  This function queries Sequence Time.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSeqTime (ViSession instrSession,
                                         ViPReal64 sequenceTime)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (!sequenceTime)   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "TS?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", sequenceTime));

    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Sig1_Sig2                                                   */
/* Purpose:  This function defines specific function for Signal1 and Signal2 */
/*           outputs on Analog Interface.  									 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSignalFunc (ViSession instrSession,
                                            ViInt32 signal1, ViInt32 signal2)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViInt32Range (signal1, 0, 9))  
      return VI_ERROR_PARAMETER2;

    if (gmsspx_invalidViInt32Range (signal2, 0, 9))  
      return VI_ERROR_PARAMETER3;

   
    if ((instrPtr -> instrModel) > 6)
      return GMSSPX_ERROR_NOSUP_FUNC_MODEL;
  

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SI %s,%s", Signal12Arr[signal1],
            Signal12Arr[signal2]));

    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Sig1_Sig2                                                   */
/* Purpose:  This function queries specific function for Signal1 and Signal2 */
/*           outputs on Analog Interface.                                    */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSignalFunc (ViSession instrSession,
                                            ViPInt32 signal1, ViPInt32 signal2)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode1[BUFFER_SIZE],mode2[BUFFER_SIZE];
    ViInt32 i;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (!signal1)  
       return VI_ERROR_PARAMETER2;

    if (!signal2)  
       return VI_ERROR_PARAMETER3;

    if ((instrPtr -> instrModel) > 6)
       return GMSSPX_ERROR_NOSUP_FUNC_MODEL;
  
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SI?"));
  
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %[^,],%[^\n]", mode1, mode2));

  for (i = 0; Signal12Arr[i]; i++)
    if (!strcmp(mode1, Signal12Arr[i])) break;

  if (!Signal12Arr[i])
    return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
  
    *signal1 = i;    
  
  i = 0;
  
  for (i = 0; Signal12Arr[i]; i++)
    if (!strcmp(mode2, Signal12Arr[i])) break;

  if (!Signal12Arr[i])
    return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
  
    *signal2 = i;    
  
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set Trigger Mode                                                */
/* Purpose:  This function defines Trigger Mode.                             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetTriggerMode (ViSession instrSession,
                                             ViInt32 mode)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViInt32Range (mode,0,5))   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "T_M %s", TriggerModeArr[mode]));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Trigger Mode                                                */
/* Purpose:  This function returns Trigger Mode.                             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetTriggerMode (ViSession instrSession,
                                             ViPInt32 triggerMode)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode[BUFFER_SIZE];
    ViInt32 i;
  gmsspx_instrRange instrPtr;
    
    if (!triggerMode)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "T_M?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  for (i = 0; TriggerModeArr[i]; i++)
    if (!strcmp(mode, TriggerModeArr[i])) break;

  if (!TriggerModeArr[i])
    return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
  
    *triggerMode = i;    
    
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set SSet                                                        */
/* Purpose:  This function sets Signal Set (SSet) functionallity in      	 */
/*           the instrument.												 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetSset (ViSession instrSession, ViBoolean state)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode1[BUFFER_SIZE],mode2[BUFFER_SIZE];
    ViInt32 i;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViBooleanRange (state))  
        return VI_ERROR_PARAMETER2;

    if ((instrPtr -> instrModel) > 6)
       return GMSSPX_ERROR_NOSUP_FUNC_MODEL;
  

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SS %s", StateArr[state]));  
  
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get SSet                                                        */
/* Purpose:  This function gets Signal Set (SSet) functionallity in 	     */
/*           the instrument.												 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetSset (ViSession instrSession, ViPBoolean state)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode1[BUFFER_SIZE],mode2[BUFFER_SIZE];
    ViInt32 i;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
   
	if (!state)  
       return VI_ERROR_PARAMETER2;
       
    if ((instrPtr -> instrModel) > 6)
       return GMSSPX_ERROR_NOSUP_FUNC_MODEL;
  

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SS?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode1));

  *state = (ViBoolean) strcmp( mode1, "OFF") ? VI_TRUE : VI_FALSE; 


  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set MinMax State                                                */
/* Purpose:  This function selects MINMAX mode.                              */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetMinMaxState (ViSession instrSession,
                                             ViBoolean state)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViBooleanRange (state))  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "MI %s", StateArr[state]));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get MinMax State                                                */
/* Purpose:  This function queries MINMAX mode.                              */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetMinMaxState (ViSession instrSession,
                                             ViPBoolean state)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;

    if (!state)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "MI?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

    *state = (ViBoolean) strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE;    

    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Set Power-On Mode                                               */
/* Purpose:  This function defines the status of the device setting after	 */
/*			 power on.   										             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confSetPwrOnMode (ViSession instrSession,
                                           ViInt32 powerMode)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViInt32Range (powerMode,0,2))  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "POW %s", PowerModeArr[powerMode]));
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Power-On Mode                                               */
/* Purpose:  This function queries the status of the device setting after	 */
/*			 power on.														 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_confGetPwrOnMode (ViSession instrSession,
                                           ViPInt32 powerMode)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
    ViChar mode[BUFFER_SIZE];
    ViInt32 i;
  gmsspx_instrRange instrPtr;
    
    if (!powerMode)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "POW?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  for (i = 0; PowerModeArr[i]; i++)
    if (!strcmp(mode, PowerModeArr[i])) break;

  if (!PowerModeArr[i])
    return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
  
    *powerMode = i;    
    
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Sequence Control                                                */
/* Purpose:  This function operates the sequence trigger.                    */                           
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_actSendSeqTrigger (ViSession instrSession,
                                            ViInt32 trigger)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
    gmsspx_instrRange instrPtr;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (gmsspx_invalidViInt32Range (trigger, 0, 5))  
        return VI_ERROR_PARAMETER2;

  if ((instrPtr -> instrModel) > 6)
    if (trigger != 0)  
           return VI_ERROR_PARAMETER2;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "SEQ %s", SequenceTriggArr[trigger]));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Reset MinMax Mode                                               */
/* Purpose:  This function resets values in MinMax memory.                   */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_resetMinMax (ViSession instrSession)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);  
  CHECKERR (viPrintf (instrSession, "MI RST"));
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       
                                        
/*===========================================================================*/
/* Function: Fetch MinMax                                                    */
/* Purpose:  This function returns the limit values of the output current and*/
/*           voltage measured and saved in MINMAX memory during MINMAX ON.   */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_fetchMinMax (ViSession instrSession,
                                      ViPReal64 minVoltage,
                                      ViPReal64 maxVoltage,
                                      ViPReal64 minCurrent,
                                      ViPReal64 maxCurrent)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));    
  
  if (minVoltage)  {
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "UMIN?"));
    Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", minVoltage));

  }
  
  if (maxVoltage) {
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "UMAX?")); 
    Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", maxVoltage)); 
  }

  if (minCurrent) {
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "IMIN?")); 
    Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", minCurrent)); 
  }

  if (maxCurrent) {
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "IMAX?")); 
    Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", maxCurrent)); 
  }

  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       
                                          
/*===========================================================================*/
/* Function: Meas Output Voltage                                             */
/* Purpose:  This function queries the actual measured value of the output   */
/*			 voltage.                										 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_measOutVolt (ViSession instrSession, ViPReal64 voltage)
{                                         
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (!voltage)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "UOUT?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", voltage));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                    

/*===========================================================================*/
/* Function: Get Output Current                                              */
/* Purpose:  This function queries the actual measured value of the output   */
/*           current.                         								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_measOutCurr (ViSession instrSession, ViPReal64 current)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
  if (!current)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "IOUT?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", current));
  
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Meas Output Power                                               */
/* Purpose:  This function queries the actually measured power on the output.*/
/*           The time required to acquire and process the reading is approx. */
/*           90 ms.                         								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_measOutPower (ViSession instrSession,
                                       ViPReal64 powerOut)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
  if (!powerOut)   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "POUT?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %Lf", powerOut));

    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Recall Register                                                 */
/* Purpose:  This function recalls the specified register from the			 */
/*           instrument's memory.											 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_dataRecallRegister (ViSession instrSession,
                                             ViInt32 registerNo)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViInt32Range (registerNo, 1, 255))   
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "*RCL %ld", registerNo));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Save Register                                                   */
/* Purpose:  This function stores settings to the register in instrument's   */
/*           memory.                 										 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_dataSaveRegister (ViSession instrSession,
                                           ViInt32 registerNo)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViInt32Range (registerNo, 1, 255))   
       return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "*SAV %ld", registerNo));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Store Register Data                                             */
/* Purpose:  This function stores parameters to register.                    */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_dataStoreReg (ViSession instrSession,
                                       ViInt32 registerNo, ViReal64 voltage,
                                       ViReal64 current,
                                       ViReal64 sequenceTime, ViInt32 state)
{                               
  ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

  if ((instrPtr -> instrModel) > 6)
    return GMSSPX_ERROR_NOSUP_FUNC_MODEL;

    if (gmsspx_invalidViInt32Range (registerNo, 11, 255))  
       return VI_ERROR_PARAMETER2;

    if (gmsspx_invalidViReal64Range (voltage, 0.0, VoltageMaxArr[instrPtr -> instrModel]))   
       return VI_ERROR_PARAMETER3;

    if (gmsspx_invalidViReal64Range (current, 0.0, CurrentMaxArr[instrPtr -> instrModel]))   
       return VI_ERROR_PARAMETER4;

    if (gmsspx_invalidViReal64Range (sequenceTime, 0.0, 99.99))  
       return VI_ERROR_PARAMETER5;

    if (gmsspx_invalidViInt32Range (state, 0, 2))  
       return VI_ERROR_PARAMETER6;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "STO %d, %.3f, %.3f, %.3f, %s",
            registerNo, voltage, current, sequenceTime, StoreStateArr[state]));
    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Read Register Data                                              */
/* Purpose:  This function returns parameters to register.                   */           
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_dataReadReg (ViSession instrSession,
                                       ViInt32 registerNo, ViPReal64 voltage,
                                       ViPReal64 current,
                                       ViPReal64 sequenceTime, ViPInt32 state)
{                               
  ViStatus	gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
  ViChar 	state_str[10];
  ViChar 	buf[BUFFER_SIZE];

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

  if ((instrPtr -> instrModel) > 6)
    return GMSSPX_ERROR_NOSUP_FUNC_MODEL;

  if (gmsspx_invalidViInt32Range (registerNo, 11, 255))  
       return VI_ERROR_PARAMETER2;
  if (!voltage)  
       return VI_ERROR_PARAMETER3;
  if (!current)  
       return VI_ERROR_PARAMETER4;
  if (!sequenceTime)
       return VI_ERROR_PARAMETER5;
  if (!state)
       return VI_ERROR_PARAMETER6;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "STO? %d", registerNo));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %*ld,%lf,%lf,%lf, %s", voltage, current, sequenceTime, state_str));
  
  CHECKERR (gmsspx_checkStatus (instrSession));
  if (!strcmp(state_str, "OFF")) {
    *state = 0;
  } else if (!strcmp(state_str, "ON")) {
    *state = 1;
  } else if (!strcmp(state_str, "CLR")) {
    *state = 2;
  }
  
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Store Registers Data Array                                      */
/* Purpose:  This function stores parameters to more registers.              */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_dataStoreRegDataArr (ViSession instrSession,
                                              ViInt32 startRegisterNo,
                                              ViReal64 _VI_FAR voltage[],
                                              ViReal64 _VI_FAR current[],
                                              ViReal64 _VI_FAR sequenceTime[],
                                              ViInt32 _VI_FAR state[],
                                              ViInt32 numberOfElements)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  ViInt32 i;

    if (!startRegisterNo)  
       return VI_ERROR_PARAMETER2;

    if (!voltage)  
       return VI_ERROR_PARAMETER3;

    if (!current)  
       return VI_ERROR_PARAMETER4;

    if (!sequenceTime)   
       return VI_ERROR_PARAMETER5;

    if (!state)  
       return VI_ERROR_PARAMETER6;
    
    if (gmsspx_invalidViInt32Range (numberOfElements, 1, 245))   
       return VI_ERROR_PARAMETER7;

  for (i=0; i < numberOfElements; i++)
  {
    CHECKERR (gmsspx_dataStoreReg (instrSession, startRegisterNo + i, voltage[i],
                        current[i], sequenceTime[i], state[i])); 
    Delay(0.1);
  }

    CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/

/*****************************************************************************/
/*-------- INSERT USER-CALLABLE INSTRUMENT-DEPENDENT ROUTINES HERE ----------*/
/*****************************************************************************/
           
/*===========================================================================*/
/* Function: Set System Register                                             */
/* Purpose:  This function sets specified value to register.                 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_utilSetReg (ViSession instrSession,
                                     ViInt32 registerName, ViInt32 value)
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

    if (gmsspx_invalidViInt32Range (registerName, 0, 4))   
       return VI_ERROR_PARAMETER2;

    if (gmsspx_invalidViInt32Range (value, 0, 255))  
       return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "%s %d", SetRegArr[registerName], value));

    CHECKERR (gmsspx_checkStatus (instrSession));

  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Register                                                    */
/* Purpose:  This function gets specified value from register.               */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_utilGetReg (ViSession instrSession,
                                     ViInt32 registerName, ViPInt32 value)
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

    if (gmsspx_invalidViInt32Range (registerName, 0, 9))   
       return VI_ERROR_PARAMETER2;

    if (!value)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "%s", GetRegArr[registerName]));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%d", value));

    CHECKERR (gmsspx_checkStatus (instrSession));

  return gmsspx_status;

}

/*===========================================================================*/
/* Function: Check State                                                     */
/* Purpose:  This function switches ON/OFF state checking of the instrument  */
/*           (reading of the Standard Event Register and checking it for     */
/*           error). Nearly all driver function are using state checking.    */
/*           Switch this function to ON when debug your application. For     */
/*           better bus throughput and instruments performance switch it OFF.*/
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_errorCheckState (ViSession instrSession, ViBoolean state)
{
    ViStatus gmsspx_status = VI_SUCCESS;
    gmsspx_instrRange instrPtr;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    instrPtr -> errorChecking = state;

    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "%s", writeBuffer));
    
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_readInstrData (ViSession instrSession, ViInt32 numberBytesToRead,
                    ViChar _VI_FAR readBuffer[], ViPInt32 numBytesRead)
{
    ViStatus  gmsspx_status = VI_SUCCESS;
    ViUInt32  retCnt;
  gmsspx_instrRange instrPtr;
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (numBytesRead)
      *numBytesRead = 0L;
        
    Delay (instrPtr->ioDelay);
    CHECKERR (viRead (instrSession, (ViBuf)readBuffer, numberBytesToRead, &retCnt));
    readBuffer[retCnt - 1] = 0;
    
    if (numBytesRead)
      *numBytesRead = (ViInt32)retCnt;
    
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.  If the reset function     */
/*           is not supported by the instrument, this function returns       */
/*           the warning VI_WARN_NSUP_RESET.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_reset (ViSession instrSession)
{
    ViStatus gmsspx_status = VI_SUCCESS;
    ViUInt32 retCnt;
    gmsspx_instrRange instrPtr;
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  
    /*  Initialize the instrument to a known state.  */
  
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "*RST"));
 
    Delay (0.2);

    CHECKERR (gmsspx_defaultInstrSetup (instrSession));
        
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result. 			                                         */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_self_test (ViSession instrSession, ViPInt16 testResult,
                    ViChar _VI_FAR testMessage[])
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;

    if (!testResult)
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "*TST?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%hd", testResult));
  
  if (testMessage)
    strcpy(testMessage, (testResult) ? "Self-test passed" : "Self-test failed");

    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Diagnostics                                                     */
/* Purpose:  This function executes the instrument diagnostics and returns   */
/*           the result. If the diagnostics function is not supported by the */
/*           instrument, this function returns error code:                   */
/*           GMSSPX_ERROR_NOSUP_FUNC_MODEL                                   */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_diag (ViSession instrSession,
                               ViChar _VI_FAR selfTestMessage[])
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
    
  if ((instrPtr -> instrModel) < 7)
    return GMSSPX_ERROR_NOSUP_FUNC_MODEL;
  
    if (!selfTestMessage)
       return VI_ERROR_PARAMETER2;

  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "HID_TST?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%[^\n]", selfTestMessage));
  
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_error_query (ViSession instrSession, ViPInt32 errorCode,
                    ViChar _VI_FAR errorMessage[])
{
  return VI_WARN_NSUP_ERROR_QUERY;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_error_message (ViSession instrSession, ViStatus statusCode,
                    ViChar _VI_FAR errMessage[])
{
    ViStatus gmsspx_status = VI_SUCCESS;
    ViInt16 i;

    static gmsspx_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,     "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,        "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,    "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,  "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,    "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE, "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
    {GMSSPX_ERROR_INSTRUMENT_ERROR,  "ERROR: Instrument status error"},
        {GMSSPX_ERROR_UNKNOWN_INSTR_MODEL, "ERROR: Unknown instrument model"},
        {GMSSPX_ERROR_NOSUP_FUNC_MODEL, "ERROR: Function not supported for this model"},  
        {GMSSPX_ERROR_INVALID_CONFIGURATION, "ERROR: Instrument configuration error"},
        {GMSSPX_ERROR_QUERY_ERROR,"ERROR: Invalid query."},
        {GMSSPX_ERROR_EXECUTION_ERROR,"ERROR: Execution error"},
        {GMSSPX_ERROR_COMMAND_ERROR,"ERROR: Command error"},
        {GMSSPX_ERROR_LIMIT_ERROR,"ERROR: Limit error"},
        {GMSSPX_ERROR_DDT_ERROR,"ERROR: Define Device Trigger error"},
        {GMSSPX_ERROR_OUT_ERROR,"ERROR: Output-On error"},
        {GMSSPX_ERROR_SEQ_ERROR,"ERROR: Sequence execution error"},
        {VI_NULL, VI_NULL}
    };

    gmsspx_status = viStatusDesc (instrSession, statusCode, errMessage);
    if (gmsspx_status == VI_WARN_UNKNOWN_STATUS) {
        for (i = 0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                sprintf (errMessage, "%s", statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        sprintf (errMessage, "Unknown Error 0x%x", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    gmsspx_status = VI_SUCCESS;
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*           If the revision query function is not supported by the          */ 
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_REV_QUERY.                                         */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_revision_query (ViSession instrSession,
                    ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);

  CHECKERR (viPrintf (instrSession, "*IDN?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^,],%*[^,],%*[^,],%[^\n]", instrRev));
  if (driverRev != NULL) sprintf (driverRev, "%s", GMSSPX_REVISION);
    
    return gmsspx_status;
}

/*===========================================================================*/
/* Function: Set Display State                                               */
/* Purpose:  This function selects display mode.                             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_utilSetDisplayState (ViSession instrSession,
                                              ViBoolean state)
{                               
    ViStatus gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;
    
    if (gmsspx_invalidViBooleanRange (state))  
       return VI_ERROR_PARAMETER2;
  
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "DI %s", StateArr[state]));
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}

/*===========================================================================*/
/* Function: Get Display State                                               */
/* Purpose:  This function queries display mode.                             */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_utilGetDisplayState (ViSession instrSession,
                                              ViPBoolean state)
{                                         
  ViStatus gmsspx_status = VI_SUCCESS;
  ViChar mode[BUFFER_SIZE];
  gmsspx_instrRange instrPtr;

  if (!state)  
       return VI_ERROR_PARAMETER2;

  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  Delay (instrPtr->ioDelay);
  CHECKERR (viPrintf (instrSession, "DI?"));
  Delay (instrPtr->ioDelay);
  CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

  *state = (ViBoolean) strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE;    
    
  CHECKERR (gmsspx_checkStatus (instrSession));
    
  return gmsspx_status;
}                                       

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmsspx_close (ViSession instrSession)
{
    gmsspx_instrRange instrPtr;
    ViSession rmSession;
    ViStatus gmsspx_status = VI_SUCCESS;

    if ((gmsspx_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
        return gmsspx_status;
    if ((gmsspx_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
        return gmsspx_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    gmsspx_status = viClose (instrSession);
    viClose (rmSession);

    return gmsspx_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean gmsspx_invalidViBooleanRange (ViBoolean val)
{
    return (ViBoolean)((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean gmsspx_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return (ViBoolean)((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean gmsspx_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return (ViBoolean)((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Check String                                                    */
/* Purpose:  This function checks a string value to see if it is             */
/*           of a defined length and initialized.                            */
/*===========================================================================*/
ViBoolean gmsspx_invalidViString (ViString val, ViUInt32 maxLength)
{
    if (val)
        {
        if (strlen (val) > maxLength)
            return VI_TRUE;
        else
            return VI_FALSE;
        }

    return VI_TRUE;
}

/*================================================================================*/
/* Function: Wait On VISA Event                                                   */
/* Purpose:  This function waits for the occurrence of an enabled VISA Event. The */
/*           event type must be enabled before entering this function. Any        */
/*           programmatic commands, actions, or conditions, necessary to generate */
/*           the specified event type, must be satisified before entering this    */
/*           function. The function will wait for the specified timeout and then  */
/*           return. If the specified event is received then the function will    */
/*           return VI_SUCCESS (0), otherwise the status code for the generated   */
/*           error will be returned. In either case the function will disable the */
/*           event type and deallocate the event handle passed from viWaitOnEvent */
/*           before returning. WARNING: If VI_TMO_INFINITE is passed in as the    */
/*           timeout this function WILL NOT return until the specified event is   */
/*           received, if the event is not received the function will not return  */
/*           and it will be necessary to terminate CVI in order to regain control.*/
/*================================================================================*/
ViStatus gmsspx_waitOnVisaEvent (ViSession instrSession, ViEventType eventTypeIn, 
            ViUInt32 timeoutIn, ViPUInt16 STB)
{            
    ViStatus gmsspx_status = VI_SUCCESS, tempStatus = VI_SUCCESS;
    ViEventType eventTypeOut = 0;
    ViEvent eventHandle = 0;
    
    /* For debug purposes we want to be able to see the status returned by        */
    /* viDisableEvent and viClose if one of the internal functions fails but do   */
    /* not want to return that value as that is not where the initial error       */
    /* occurs in the function, so we assign it to tempStatus.                     */
    
    if ((gmsspx_status = viWaitOnEvent (instrSession, eventTypeIn, timeoutIn, 
                                        &eventTypeOut, &eventHandle)) < 0) {
        tempStatus = viDisableEvent (instrSession, eventTypeIn, VI_QUEUE);
        return gmsspx_status;
    } 
    
    if (eventTypeIn == VI_EVENT_SERVICE_REQ)
        if ((gmsspx_status = viReadSTB (instrSession, STB)) < 0) {
            tempStatus = viClose (eventHandle);
            tempStatus = viDisableEvent (instrSession, eventTypeIn, VI_QUEUE);
            return gmsspx_status;
        }   
        
    if ((gmsspx_status = viClose (eventHandle)) < 0) {             
        tempStatus = viDisableEvent (instrSession, eventTypeIn, VI_QUEUE);
        return gmsspx_status;                 
    }
    
    if ((gmsspx_status = viDisableEvent (instrSession, eventTypeIn, VI_QUEUE)) < 0)
        return gmsspx_status;                                                   

    return gmsspx_status;                                    
}    

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the gmsspx_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from gmsspx_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus gmsspx_initCleanUp (ViSession openRMSession,
                    ViPSession openInstrSession, ViStatus currentStatus)
{
    gmsspx_instrRange instrPtr;
    
    if (viGetAttribute (*openInstrSession, VI_ATTR_USER_DATA, &instrPtr) >= 0)
        if (instrPtr != NULL) 
            free (instrPtr);

    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*****************************************************************************/
/*----------- INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINES HERE -------------*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the gmsspx_reset operation and by the     */
/*           gmsspx_init function if the reset option has not been           */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus gmsspx_defaultInstrSetup (ViSession instrSession)
{
    ViStatus gmsspx_status = VI_SUCCESS;
    gmsspx_instrRange instrPtr;
    ViChar instrM[BUFFER_SIZE], buf[BUFFER_SIZE];
    ViUInt32 i, retCnt;
    ViPChar model;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    instrPtr -> instrModel = 0;
    instrPtr -> errorChecking = VI_TRUE;
    instrPtr->ioDelay = DEFAULT_IO_DELAY; // 2 ms
    strcpy (instrPtr -> instrDriverRevision, GMSSPX_REVISION);

    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "*IDN?"));
    Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%*[^,],%[^,]", instrM));

    if (*instrM == 'S') {
      memmove (instrM, instrM+3, 11);
    }
    instrM[11]=0;

    for (i = 0; model = InstrumentModelArr[i]; i++)
      if (!strcmp(instrM, model)) break;
    
    if (model == VI_NULL)
      return GMSSPX_ERROR_UNKNOWN_INSTR_MODEL;

    instrPtr -> instrModel = i;
    
    Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "*ESE 52")); 
    Delay (instrPtr->ioDelay);
    
    if (i < 7) {
      CHECKERR (viPrintf (instrSession, "ERBE 60"));
    } else {
      CHECKERR (viPrintf (instrSession, "ERBE 58"));
    }

    return gmsspx_status;
}

/*=========================================================================*/
/* Function: Check Status                                                  */
/* Purpose:  This function reads the system status byte.                   */
/*=========================================================================*/
ViStatus gmsspx_checkStatus (ViSession instrSession)
{
    ViStatus gmsspx_status = VI_SUCCESS;
    ViInt32 stbByte = 0;
    gmsspx_instrRange instrPtr;
    ViChar buf[BUFFER_SIZE];
    ViUInt32 cnt, retCnt;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (instrPtr -> errorChecking) { /* Is Error Checking Enabled? */
        /*--------------- Read the Standard Event Register --------------------*/
        Delay (instrPtr->ioDelay);
        CHECKERR (viWrite (instrSession, "*STB?", 5, &retCnt));
        Delay (instrPtr->ioDelay);
        CHECKERR (viRead (instrSession, buf, BUFFER_SIZE, &retCnt));
        buf[retCnt - 1] = 0;
        if (sscanf (buf, "%ld", &stbByte) != 1)
          return GMSSPX_ERROR_INSTRUMENT_ERROR;
        
        
        /*---------------- Check if any error or message bit was asserted -------*/
        if (stbByte & 32) { // ESR
          ViInt32 esrByte = 0;
          
          Delay (instrPtr->ioDelay);
          CHECKERR (viWrite (instrSession, "*ESR?", 5, &retCnt));
          Delay (instrPtr->ioDelay);
          CHECKERR (viRead (instrSession, buf, BUFFER_SIZE, &retCnt));
          buf[retCnt - 1] = 0;
          if (sscanf (buf, "%ld", &esrByte) != 1)
            return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
          
          if (esrByte & 4) return GMSSPX_ERROR_QUERY_ERROR;
          if (esrByte & 16) return GMSSPX_ERROR_EXECUTION_ERROR;
          if (esrByte & 32) return GMSSPX_ERROR_COMMAND_ERROR;
        }
        if (stbByte & 4) { // ERB
          ViInt32 erbByte = 0;
          int limerr;
          if ((instrPtr -> instrModel) < 7) limerr = 4; else limerr = 2;
          Delay (instrPtr->ioDelay);
          CHECKERR (viWrite (instrSession, "ERB?", 4, &retCnt));
          Delay (instrPtr->ioDelay);
          CHECKERR (viRead (instrSession, buf, BUFFER_SIZE, &retCnt));
          buf[retCnt - 1] = 0;
          if (sscanf (buf, "%ld", &erbByte) != 1) {
            return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
          }
          if (erbByte & limerr) return GMSSPX_ERROR_LIMIT_ERROR;
          if (erbByte & 8) return GMSSPX_ERROR_DDT_ERROR;
          if (erbByte & 16) return GMSSPX_ERROR_OUT_ERROR;
          if (erbByte & 32) return GMSSPX_ERROR_SEQ_ERROR;
        }
    }

    return gmsspx_status;
}

/*=========================================================================*/
/* Function: Get Instrument Model                                          */
/* Purpose:  This function returns the instrument model.                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_utilGetInstrModel (ViSession instrSession,
                							ViChar _VI_FAR instrumentModel[],
                							ViPInt32 instrModelNumber)
{
  ViStatus			gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange instrPtr;    
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  if (instrModelNumber)
    *instrModelNumber = instrPtr -> instrModel;
  strcpy (instrumentModel, InstrumentModelArr[instrPtr -> instrModel]);
    
  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Set I/O Delay Time                                            */
/* Purpose:  This function set a delay between each two instrument I/O     */
/*           operations.                                                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_utilSetIODelay (ViSession instrSession,
                						 ViReal64 IODelayTime)
{
  ViStatus			gmsspx_status = VI_SUCCESS;
  gmsspx_instrRange	instrPtr;    
  
  if (IODelayTime < 0.0)  
       return VI_ERROR_PARAMETER2;
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  instrPtr->ioDelay = IODelayTime;
    
  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Get I/O Delay Time                                            */
/* Purpose:  This function returns a delay between each two instrument I/O */
/*           operations.                                                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_utilGetIODelay (ViSession instrSession,
                						 ViPReal64 IODelayTime)
{
	ViStatus			gmsspx_status = VI_SUCCESS;
	gmsspx_instrRange	instrPtr;    
  
  if (!IODelayTime)  
       return VI_ERROR_PARAMETER2;
    
  CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

   *IODelayTime = instrPtr->ioDelay;
    
  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Address                                                       */
/* Purpose:  This function address the appropriate instrument when several */
/* 			 instruments are chained via serial cable.                     */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_utilAddress(ViSession instrSession, ViInt32 address)
{

ViStatus gmsspx_status = VI_SUCCESS;
 
  if (gmsspx_invalidViInt32Range (address, 1, 32))  
       return VI_ERROR_PARAMETER2;
  
  CHECKERR (viPrintf (instrSession, "ADDR %d", address));

  return gmsspx_status;
}

/*=========================================================================*/
/* Function: Lock Instrument                                               */
/* Purpose:  This function locks/unlocks the instrument.                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmsspx_utilLockInstr (ViSession instrSession, 
										ViBoolean instrument)
{
ViStatus gmsspx_status = VI_SUCCESS;
 
  if (gmsspx_invalidViBooleanRange (instrument))  
       return VI_ERROR_PARAMETER2;
  
  if (instrument)
	 gmsspx_status = viGpibControlREN (instrSession, VI_GPIB_REN_ASSERT_LLO); //lock
  else
  	gmsspx_status = viGpibControlREN (instrSession, VI_GPIB_REN_DEASSERT_GTL); //unlock  
  
  return gmsspx_status;
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
